<h3>What Does This Do?</h3>
<p>There are often times on websites where tasks that may take a considerable amount of time need to be automated.  For example, sending out an email newsletter, importing records from a CSV file, or requesting large amounts of data from a remote service.   PHP (perticularly on hosted websites) does not like to run for long periods of time.  This module allows a developer to create &quot;jobs&quot; consisting of one or more &quot;tasks&quot; that are handled by an http request that can be handled in the background automatically.<p>
<h3>How do I use it</h3>
<p>This module provides an API that allows other module developers to create jobs.  From a user perspective you merely need to follow two steps:</p>
<ol>
  <li>Install and configure the module.</li>
  <li>Configure a scheduled task to access the supplied process url on a regular, frequent basis <em>(a cron job works great for this)</em>
</ol>
<h3>Features</h3>
<ul>
  <li>The API allows for creating startup and finish jobs...</li>
  <li>An iterative task type that will run small steps (such as converting one line of a CSV file) until the allowed time is nearly exhausted</li>
  <li>A Simple task type that runs once and is expected to complete.</li>
  <li>Extensive logging to the admin log</lI>
  <li>Jobs are started by the process action (accessed via the process url provided in the admin interface, one job per request) then complete until finishede via continued redirectsw back to the process action.  Therefore jobs will complete faster.</li>
  <li>Small amounts of data can be attached to the job, and to individual tasks that is serialized and stored with the job in the database.  This is useful for providing information such as filenames, time limits, notification addresses etc. to the task functions.</li>
</ul>
<h3>Automatic Processing</h3>
<p>Though the CGJobMgr module provides a link in its admin panel to process the next job, it is best if this is done automatically on a frequent basis.  There are numerous ways that this can be done.</p>
<p>Most webhosts provide a control panel that has an option for creating and managing scheduled tasks (stuff that you want to do on a regular basis).  If not, and you have access to a task manager on another machine that is internet enabled you can create a scheduled task there.</p>
<p>Linux/Unix based machines have a built in mechanism called &quot;cron&quot; Below is a sample job that is executed every 10 minutes, and uses wget to trigger the processing and disregard its output.<br/><br/></p>
<code style="margin-left: 4em; margin-right: 4em; background-color: white; border: 1px solid black; padding: 1em;">
*/10 * * * * wget -O /dev/null http://mysite.com/CGJobMgr/process >/dev/null 2>&1
</code>
<h3>API</h3>
<p>Numerous PHP classes define the API for creating jobs, their tasks, and then submitting them to the job manager.   The classes are fully documented in their appropriate PHP files.  The files are as follows:</p>
<table class="pagetable" cellspacing="0" style="margin-left: 2em; width: 75%;">
 <thead>
  <tr>
    <th>Class</th>
    <th>Description</th>
    <th>Filename</th>
  </tr>
 </thead>
 <tbody>
   <tr>
     <td>cgjobmgr_job</td>
     <td>The job class</td>
     <td>lib/class.cgjobmgr_job.php</td>
   </tr>
   <tr>
     <td>cgjobmgr_task</td>
     <td>Abstract base class</td>
     <td>lib/class.cgjobmgr_task.php</td>
   </tr>
   <tr>
     <td>cgjobmgr_simpletask</td>
     <td>A simple &quot;run once&quot; task</td>
     <td>lib/class.cgjobmgr_simpletask.php</td>
   </tr>
   <tr>
     <td>cgjobmgr_iterativetask</td>
     <td>A task that will run small steps (i.e: process one line of a CSV file) until allotted time is close to expiry.</td>
     <td>lib/class.cgjobmgr_iterativetask.php</td>
   </tr>
   <tr>
     <td>cgjobmgr_jobhandler</td>
     <td>An abstract singleton class that provides the interface for different job handlers.</td>
     <td>lib/class.cgjobmgr_jobhandler.php</td>
   </tr>
 </tbody>
</table>
<h3>Example:</h3>
<p>The following code is a marked up copy of the code from the CGRapLeaf module that submits the job.</p>
<pre style="margin-left: 4em; margin-right: 4em; border: 1px solid black; background-color: white;"><code>// Create a new job, give it a name, and specify who owns it
$job = new cgjobmgr_job($this->GetName().' bulk submission',get_userid(FALSE));

// Create a new &quot;iterative&quot; task that will repeat a small step until there is no time left
$task = new cgjobmgr_iterativetask('process email address');

// Specify which function this task will call to actually do the work
$task->set_function(array('cgrapleaf_task','process_task'));

// Provide some data to the task
$task->set_persistent_data('input_file',$infile);
$task->set_persistent_data('output_file',$outfile);
$task->set_persistent_data('logfile',$logfile);
$task->set_persistent_data('column',$column);
$task->set_persistent_data('delimeter',$delimiter);

// Add the task to the job
$job->add_task($task);

// Create another task... a simple run-once task
$task = new cgjobmgr_simpletask('send result');
$task->set_function(array('cgrapleaf_task','notify'));

// Add some data into this task
$task->set_persistent_data('input_file',$infile);
$task->set_persistent_data('output_file',$outfile);
$task->set_persistent_data('logfile',$logfile);
$task->set_persistent_data('dest_email',$email);

// Add the task to the job
$job->add_task($task);

// Save the job into the database... it will run automatically the next time the procesing url is accessed.
$job->save();
</code></pre><br/>
<h4>Task functions:</h4>
<p>Task functions may be simple PHP functions, or static members of a php class.  They accept a cgjobmgr_task reference as their single parameter.  They are expected to return a cgjobmgr_task status.  i.e:</p>
<pre style="margin-left: 4em; margin-right: 4em; border: 1px solid black; background-color: white;"><code>
class cgrapleaf_task
{
  public static function notify(cgjobmgr_task& $task)
  {
    $dest_email = $task->get_persistent_data('dest_email');
    $outfile = $task->get_persistent_data('output_file');

    if( !$dest_email )
      {
	audit('','cgrapleaf_task','No destination email specified');
	return cgjobmgr_task::STATUS_ERROR;
      }
    $to_addrs = cge_array::smart_explode($dest_email);
    if( !is_array($to_addrs) || count($to_addrs) == 0 )
      {
	audit('','cgrapleaf_task','Could not find a destination email address');
	return cgjobmgr_task::STATUS_ERROR;
      }

    if( !$outfile || !file_exists($outfile) )
      {
	audit('','cgrapleaf_task','Could not find output file');
	return cgjobmgr_task::STATUS_ERROR;
      }
    $outtype = cge_utils::get_mime_type($outfile);
    $rapleaf = cms_utils::get_module('CGRapLeaf');
    $cmsmailer = cms_utils::get_module('CMSMailer');
    $cmsmailer->reset();
    $num_to = 0;
    foreach( $to_addrs as $one )
      {
	if( !is_email($one) ) continue;
	$num_to++;
	$cmsmailer->AddAddress($one);
      }
    $cmsmailer->SetSubject($rapleaf->Lang('bulkemail_subject'));
    $cmsmailer->AddAttachment( $outfile, 'rapleaf.csv', 'base64', $outtype );
    if( $logfile && file_exists($logfile) )
      {
	$cmsmailer->AddAttachment( $logfile, 'rapleaf.log', 'base64', cge_utils::get_mime_type($logfile));
      }
    // now the body.
    $body = 'Your CGRapLeaf Job has completed.  The output file(s) are attached to this email';
    $cmsmailer->SetBody($body);
    $cmsmailer->Send();

    audit('','cgrapleaf_task',sprintf('Sent email to %d addresses',count($num_to)));
    return cgjobmgr_task::STATUS_COMPLETE;

  }
}
</code></pre>
<h3>Future versions</h3>
<ul>
  <li>Foreground jobs - The ability to process the job in the foreground with a progress bar and feedback.</li>
  <li>More task types</li>
  <li>Better reporting - perhaps some email notification on failed jobs, etc.</li>
  <li>Improved admin interface - The ability to resume a failed job, suspend jobs for later...</li>
  <li>
</ul>
<h3>Support</h3>
<p>This module does not include commercial support. However, there are a number of resources available to help you with it:</p>
<ul>
<li>For the latest version of this module, FAQs, or to file a Bug Report or buy commercial support, please visit calguy\'s
module homepage at <a href="http://calguy1000.com">calguy1000.com</a>.</li>
<li>Additional discussion of this module may also be found in the <a href="http://forum.cmsmadesimple.org">CMS Made Simple Forums</a>.</li>
<li>The author, calguy1000, can often be found in the <a href="irc://irc.freenode.net/#cms">CMS IRC Channel</a>.</li>
<li>Lastly, you may have some success emailing the author directly.</li>  
</ul>
<h3>Copyright and License</h3>
<p>Copyright &copy; 2011, Robert Campbel <a href="mailto:calguy1000@cmsmadesimple.org">&lt;calguy1000@cmsmadesimple.org&gt;</a>. All Rights Are Reserved.</p>
<p>This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.</p>
<p>However, as a special exception to the GPL, this software is distributed
as an addon module to CMS Made Simple.  You may not use this software
in any Non GPL version of CMS Made simple, or in any version of CMS
Made simple that does not indicate clearly and obviously in its admin 
section that the site was built with CMS Made simple.</p>
<p>This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
Or read it <a href="http://www.gnu.org/licenses/licenses.html#GPL">online</a></p>
